<?php
/*
 +--------------------------------------------------------------------+
 | Copyright CiviCRM LLC. All rights reserved.                        |
 |                                                                    |
 | This work is published under the GNU AGPLv3 license with some      |
 | permitted exceptions and without any warranty. For full license    |
 | and copyright information, see https://civicrm.org/licensing       |
 +--------------------------------------------------------------------+
 */

/**
 *
 * @package CRM
 * @copyright CiviCRM LLC https://civicrm.org/licensing
 */
class CRM_Mailing_Event_BAO_MailingEventTrackableURLOpen extends CRM_Mailing_Event_DAO_MailingEventTrackableURLOpen {

  /**
   * Track a click-through and return the URL to redirect.
   *
   * If the numbers don't match up, return the base url.
   *
   * @param int $queue_id
   *   The Queue Event ID of the clicker.
   * @param int $url_id
   *   The ID of the trackable URL.
   *
   * @return string
   *   The redirection url, or base url on failure.
   */
  public static function track($queue_id, $url_id) {
    // To find the url, we also join on the queue and job tables.  This
    // prevents foreign key violations.
    $job = CRM_Utils_Type::escape(CRM_Mailing_BAO_MailingJob::getTableName(), 'MysqlColumnNameOrAlias');
    $eq = CRM_Utils_Type::escape(CRM_Mailing_Event_BAO_MailingEventQueue::getTableName(), 'MysqlColumnNameOrAlias');
    $turl = CRM_Utils_Type::escape(CRM_Mailing_BAO_MailingTrackableURL::getTableName(), 'MysqlColumnNameOrAlias');

    $urlSearch = CRM_Core_DAO::executeQuery(
      "SELECT url
        FROM $turl
        WHERE $turl.id = %1",
      [
        1 => [$url_id, 'Integer'],
      ]
    );
    if (!$urlSearch->fetch()) {
      return CRM_Utils_System::baseURL();
    }

    if (!$queue_id) {
      return $urlSearch->url;
    }

    $queueSearch = CRM_Core_DAO::executeQuery(
      "SELECT $turl.url as url
         FROM $turl
        INNER JOIN $eq ON $turl.mailing_id = $eq.mailing_id
        WHERE $eq.id = %1 AND $turl.id = %2",
      [
        1 => [$queue_id, 'Integer'],
        2 => [$url_id, 'Integer'],
      ]
    );

    if (!$queueSearch->fetch()) {
      // Can't find the queue.
      // return the URL without tracking.
      return $urlSearch->url;
    }

    self::recordUrlClick($queue_id, $url_id, date('YmdHis'));

    return $urlSearch->url;
  }

  /**
   * Get row count for the event selector.
   *
   * @param int $mailing_id
   *   ID of the mailing.
   * @param int $job_id
   *   Optional ID of a job to filter on.
   * @param bool $is_distinct
   *   Group by queue ID?.
   * @param int $url_id
   *   Optional ID of a url to filter on.
   *
   * @param string $toDate
   *
   * @return int
   *   Number of rows in result set
   */
  public static function getTotalCount(
    $mailing_id, $job_id = NULL,
    $is_distinct = FALSE, $url_id = NULL, $toDate = NULL
  ) {
    $dao = new CRM_Core_DAO();

    $click = self::getTableName();
    $queue = CRM_Mailing_Event_BAO_MailingEventQueue::getTableName();
    $mailing = CRM_Mailing_BAO_Mailing::getTableName();
    $job = CRM_Mailing_BAO_MailingJob::getTableName();

    $distinct = NULL;
    if ($is_distinct) {
      $distinct = 'DISTINCT ';
    }
    $query = "
            SELECT      COUNT($distinct $click.event_queue_id) as opened
            FROM        $click
            INNER JOIN  $queue
                    ON  $click.event_queue_id = $queue.id
            INNER JOIN  $job
                    ON  $queue.job_id = $job.id
            INNER JOIN  $mailing
                    ON  $job.mailing_id = $mailing.id
                    AND $job.is_test = 0
            WHERE       $mailing.id = " . CRM_Utils_Type::escape($mailing_id, 'Integer');

    if (!empty($toDate)) {
      $query .= " AND $click.time_stamp <= $toDate";
    }

    if (!empty($job_id)) {
      $query .= " AND $job.id = " . CRM_Utils_Type::escape($job_id, 'Integer');
    }

    if (!empty($url_id)) {
      $query .= " AND $click.trackable_url_id = " . CRM_Utils_Type::escape($url_id, 'Integer');
    }

    // query was missing
    $dao->query($query);

    if ($dao->fetch()) {
      return $dao->opened;
    }

    return NULL;
  }

  /**
   * Get tracked url count for each mailing for a given set of mailing IDs.
   *
   * @see https://issues.civicrm.org/jira/browse/CRM-12814
   *
   * @param array $mailingIDs
   *
   * @return array
   *   trackable url count per mailing ID
   */
  public static function getMailingTotalCount($mailingIDs) {
    $dao = new CRM_Core_DAO();
    $clickCount = [];

    $click = self::getTableName();
    $queue = CRM_Mailing_Event_BAO_MailingEventQueue::getTableName();
    $job = CRM_Mailing_BAO_MailingJob::getTableName();
    $mailingIDs = implode(',', $mailingIDs);

    $query = "
      SELECT $job.mailing_id as mailingID, COUNT($click.id) as opened
      FROM $click
      INNER JOIN $queue
        ON  $click.event_queue_id = $queue.id
      INNER JOIN $job
        ON  $queue.job_id = $job.id
        AND $job.is_test = 0
      WHERE $job.mailing_id IN ({$mailingIDs})
      GROUP BY civicrm_mailing_job.mailing_id
    ";

    $dao->query($query);

    while ($dao->fetch()) {
      $clickCount[$dao->mailingID] = $dao->opened;
    }
    return $clickCount;
  }

  /**
   * Get tracked url count for each mailing for a given set of mailing IDs.
   *
   * @param int[] $mailingIDs
   *   IDs of the mailing (comma separated).
   * @param int $contactID
   *   ID of the contact.
   *
   * @return array
   *   Count per mailing ID
   */
  public static function getMailingContactCount($mailingIDs, $contactID) {
    $dao = new CRM_Core_DAO();
    $clickCount = [];

    $click = self::getTableName();
    $queue = CRM_Mailing_Event_BAO_MailingEventQueue::getTableName();
    $job = CRM_Mailing_BAO_MailingJob::getTableName();
    $mailingIDs = implode(',', $mailingIDs);

    $query = "
      SELECT $job.mailing_id as mailingID, COUNT($click.id) as opened
      FROM $click
      INNER JOIN $queue
        ON  $click.event_queue_id = $queue.id
        AND $queue.contact_id = $contactID
      INNER JOIN $job
        ON  $queue.job_id = $job.id
        AND $job.is_test = 0
      WHERE $job.mailing_id IN ({$mailingIDs})
      GROUP BY civicrm_mailing_job.mailing_id
    ";

    $dao->query($query);

    while ($dao->fetch()) {
      $clickCount[$dao->mailingID] = $dao->opened;
    }

    return $clickCount;
  }

  /**
   * Get rows for the event browser.
   *
   * @param int $mailing_id
   *   ID of the mailing.
   * @param int $job_id
   *   Optional ID of the job.
   * @param bool $is_distinct
   *   Group by queue id?.
   * @param int $url_id
   *   Optional ID of a trackable URL to filter on.
   * @param int $offset
   *   Offset.
   * @param int $rowCount
   *   Number of rows.
   * @param array $sort
   *   Sort array.
   * @param int $contact_id
   *   Optional contact ID.
   *
   * @return array
   *   Result set
   */
  public static function &getRows(
    $mailing_id, $job_id,
    $is_distinct, $url_id,
    $offset = NULL, $rowCount = NULL, $sort = NULL, $contact_id = NULL
  ) {

    $dao = new CRM_Core_DAO();

    $click = self::getTableName();
    $url = CRM_Mailing_BAO_MailingTrackableURL::getTableName();
    $queue = CRM_Mailing_Event_BAO_MailingEventQueue::getTableName();
    $mailing = CRM_Mailing_BAO_Mailing::getTableName();
    $job = CRM_Mailing_BAO_MailingJob::getTableName();
    $contact = CRM_Contact_BAO_Contact::getTableName();
    $email = CRM_Core_BAO_Email::getTableName();

    $query = "
            SELECT      $contact.display_name as display_name,
                        $contact.id as contact_id,
                        $email.email as email,";

    if ($is_distinct) {
      $query .= "MIN($click.time_stamp) as date,";
    }
    else {
      $query .= "$click.time_stamp as date,";
    }

    $query .= "$url.url as url
            FROM        $contact
            INNER JOIN  $queue
                    ON  $queue.contact_id = $contact.id
            INNER JOIN  $email
                    ON  $queue.email_id = $email.id
            INNER JOIN  $click
                    ON  $click.event_queue_id = $queue.id
            INNER JOIN  $url
                    ON  $click.trackable_url_id = $url.id
            INNER JOIN  $job
                    ON  $queue.job_id = $job.id
            INNER JOIN  $mailing
                    ON  $job.mailing_id = $mailing.id
                    AND $job.is_test = 0
            WHERE       $mailing.id = " . CRM_Utils_Type::escape($mailing_id, 'Integer');

    if (!empty($contact_id)) {
      $query .= " AND $contact.id = " . CRM_Utils_Type::escape($contact_id, 'Integer');
    }

    if (!empty($job_id)) {
      $query .= " AND $job.id = " . CRM_Utils_Type::escape($job_id, 'Integer');
    }

    if (!empty($url_id)) {
      $query .= " AND $url.id = " . CRM_Utils_Type::escape($url_id, 'Integer');
    }

    if ($is_distinct) {
      $query .= " GROUP BY $queue.id, $url.url ";
    }

    $orderBy = "sort_name ASC, {$click}.time_stamp DESC";
    if ($sort) {
      if (is_string($sort)) {
        $sort = CRM_Utils_Type::escape($sort, 'String');
        $orderBy = $sort;
      }
      else {
        $orderBy = trim($sort->orderBy());
      }
    }

    $query .= " ORDER BY {$orderBy} ";

    if ($offset || $rowCount) {
      //Added "||$rowCount" to avoid displaying all records on first page
      $query .= ' LIMIT ' . CRM_Utils_Type::escape($offset, 'Integer') . ', ' . CRM_Utils_Type::escape($rowCount, 'Integer');
    }
    CRM_Core_DAO::disableFullGroupByMode();
    $dao->query($query);
    CRM_Core_DAO::reenableFullGroupByMode();
    $results = [];

    while ($dao->fetch()) {
      $url = CRM_Utils_System::url('civicrm/contact/view',
        "reset=1&cid={$dao->contact_id}"
      );
      $results[] = [
        'name' => "<a href=\"$url\">{$dao->display_name}</a>",
        'email' => $dao->email,
        'url' => $dao->url,
        'date' => CRM_Utils_Date::customFormat($dao->date),
      ];
    }
    return $results;
  }

  public static function queuedTrack(\CRM_Queue_TaskContext $ctx, $queue_id, $url_id, $dateTime) {
    $queueCheck = CRM_Core_DAO::singleValueQuery("SELECT u.url as url
      FROM civicrm_mailing_trackable_url u
      INNER JOIN civicrm_mailing_event_queue q ON u.mailing_id = q.mailing_id
      WHERE q.id = %1 AND u.id = %2", [
        1 => [$queue_id, 'Positive'],
        2 => [$url_id, 'Positive'],
      ]);
    if ($queueCheck) {
      self::recordUrlClick($queue_id, $url_id, $dateTime);
    }
  }

  private static function recordUrlClick($queue_id, $url_id, $dateTime) {
    self::writeRecord([
      'event_queue_id' => $queue_id,
      'trackable_url_id' => $url_id,
      'time_stamp' => $dateTime,
    ]);
  }

}
